/* 
 Copyright (c) 2010, NHIN Direct Project
 All rights reserved.
 
 Modified for use with the Direct Project Innovation Initiative - 2013

 Authors:
 Greg Meyer      gm2552@cerner.com
                 Bell_Adam@bah.com

 Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

 Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
 Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer 
 in the documentation and/or other materials provided with the distribution.  Neither the name of the The NHIN Direct Project (nhindirect.org). 
 nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
 BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE 
 GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, 
 STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
 THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.nhindirect.gateway.smtp.james.mailet;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Enumeration;
import java.sql.*;

import javax.mail.Header;
import javax.mail.Address;
import javax.mail.MessagingException;
import javax.mail.internet.InternetAddress;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.mailet.Mail;
import org.apache.mailet.MailAddress;
import org.apache.mailet.base.GenericMailet;
import org.nhindirect.gateway.smtp.GatewayState;
import org.nhindirect.gateway.smtp.SmtpAgent;
import org.nhindirect.gateway.smtp.SmtpAgentFactory;
import org.nhindirect.gateway.smtp.config.SmtpAgentConfig;
import org.nhindirect.stagent.AddressSource;
import org.nhindirect.stagent.NHINDAddress;

import com.google.inject.Module;
import com.google.inject.Provider;

import java.util.Locale;
import java.util.Map;

import javax.mail.internet.MimeMessage;


/**
 * Apache James mailet for logging outgoing message from local senders
 *
 * @author          
 */
public class LocalMailet extends GenericMailet {

    private static final Log LOGGER = LogFactory.getFactory().getInstance(LocalMailet.class);
    protected SmtpAgent agent;

    /**
     * {@inheritDoc}
     */
    @Override
    public void init() throws MessagingException {
        System.setProperty("enableCBCProtection", "false");
        LOGGER.info("Initializing IncomingLocalMailet");

        // Get the configuration URL
        String configURLParam = getInitParameter("ConfigURL");
        if (configURLParam == null || configURLParam.isEmpty()) {
            LOGGER.error("IncomingLocalMailet Configuration URL cannot be empty or null.");
            throw new MessagingException("IncomingLocalMailet Configuration URL cannot be empty or null.");
        }

        //parse into a URL and validate it is properly formed
        URL configURL = null;
        try {
            configURL = new URL(configURLParam);
        } catch (MalformedURLException ex) {
            LOGGER.error("Invalid configuration URL");
            throw new MessagingException("Incoming Configuration URL cannot be empty or null.", ex);
        }
        LOGGER.info("IncomingLocalMailet initialization complete.");
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    @Override
    public void service(Mail mail) throws MessagingException {
        GatewayState.getInstance().lockForProcessing();
        try {
            LOGGER.trace("Entering service(Mail mail)");

            int notificationFlag = 0;

            // Check if notification and set flag            
            try {
                if (mail.getSender().toString().equals(this.getMailetContext().getPostmaster().toString())) {
                    notificationFlag = 1;
                }
                Enumeration headers = mail.getMessage().getAllHeaders();
                while (headers.hasMoreElements()) {
                    Header h = (Header) headers.nextElement();
                    if (h.getName().contains("DirectNotification")) {
                        notificationFlag = 1;
                    }
                }
            } catch (MessagingException e) {
                LOGGER.debug("Messaging execption while trying to check notification and set flag.");
            }

            if (notificationFlag == 0) {
                // get the sender
                InternetAddress senderAddr = NHINDSecurityAndTrustMailet.getSender(mail);
                if (senderAddr == null) {
                    throw new MessagingException("Failed to process message.  The sender cannot be null or empty.");
                }
                // not the best way to do this
                NHINDAddress sender = new NHINDAddress(senderAddr, AddressSource.From);

                // Get system properties
                Map<String, String> properties = MailetProperties.getPropertiesList();

                // Database connection information
                String db_hostname = properties.get("mailet.db.hostname");
                String db_port = properties.get("mailet.db.port");
                String db_apiname = properties.get("mailet.db.apiname");
                String db_webname = properties.get("mailet.db.webname");
                String db_instance = properties.get("mailet.db.instance");
                String db_apiuserid = properties.get("mailet.db.apiusername");
                String db_webuserid = properties.get("mailet.db.webusername");
                String db_apipassword = properties.get("mailet.db.apipassword");
                String db_webpassword = properties.get("mailet.db.webpassword");

                // Retrieve support email address
                String notificationSender = properties.get("mailet.notification.sender");

                //don't log anything from the support email address
                if (!sender.toString().toLowerCase(Locale.getDefault()).equals(notificationSender.toLowerCase(Locale.getDefault()))) {
                    Connection connapi = null;
                    Connection connweb = null;
                    LOGGER.info("Logging local outgoing message from sender " + sender.toString());
                    try {
                        // DB Connection query/response setup
                        connapi = databaseConnect(db_hostname, db_port, db_apiname, db_instance, db_apiuserid, db_apipassword);
                        connweb = databaseConnect(db_hostname, db_port, db_webname, db_instance, db_webuserid, db_webpassword);

                        int log_success = 1; //since it is local, assume success until something bad happens
                        if(hasOnlyLocalRecipients(mail)){
                            LogMessage.logMessageToLog(mail, connweb, 1, 0, -1); //log incoming message to webmail, assume success
                            LogMessage.logMessageToLog(mail, connapi, 1, 0, 0); //log incoming message to API, assume success
                        }
                        ArrayList<MailAddress> badaddress = LogMessage.sendExternalNotification(connweb, mail);
                        //if we know one of the users doesn't exist, set success to false
                        if (!badaddress.isEmpty()) {
                            log_success = 0;
                        }
                        LogMessage.logMessageToLog(mail, connweb, log_success, 1, -1); //log outgoing message to webmail
                        LogMessage.logMessageToLog(mail, connapi, log_success, 1, 0); //log outgoing message to API 
                        LogMessage.proccessMessagesStatus(mail, badaddress);
                    } catch (SQLException sql_ex) {
                        LOGGER.trace("SQL Exception while sending external notifications and logging messages.");
                    } finally {
                    	try {
                            if (connweb != null) {
                                connweb.close();
                            }
                        } catch (SQLException closeEx) {
                            LOGGER.debug("Failed to close SQL Database connection.");
                        }
                        try {
                            if (connapi != null) {
                                connapi.close();
                            }
                        } catch (SQLException closeEx) {
                            LOGGER.debug("Failed to close SQL Database connection.");
                        }

                        
                    }

                }

            }
        } catch (Exception e) {
            // catch all
            LOGGER.error("Failed to log message");
        } finally {
            GatewayState.getInstance().unlockFromProcessing();
        }
    }

    /**
     * Gets a collection of Guice {@link @Module Modules} to create custom
     * bindings for Agent creation. If this method returns null, the Mailet will
     * use the simple {@link SmtpAgentFactory#createAgent(URL)} method to create
     * the {@link SmtpAgent) and
     * {@link NHINDAgent}.
     *
     * @return A collection of Guice Modules.
     */
    protected Collection<Module> getInitModules() {
        return null;
    }

    /**
     * Gets a custom configuration provider. If this is null, the system will us
     * a default provider.
     *
     * @return Gets a custom configuration provider.
     */
    protected Provider<SmtpAgentConfig> getConfigProvider() {
        return null;
    }

    public static InternetAddress getSender(Mail mail) {
        InternetAddress retVal;

        if (mail.getSender() != null) {
            retVal = mail.getSender().toInternetAddress();
        } else {
            // try to get the sender from the message
            Address[] senderAddr;
            try {
                if (mail.getMessage() == null) {
                    return null;
                }

                senderAddr = mail.getMessage().getFrom();
                if (senderAddr == null || senderAddr.length == 0) {
                    return null;
                }
            } catch (MessagingException e) {
                return null;
            }

            // not the best way to do this
            retVal = (InternetAddress) senderAddr[0];
        }

        return retVal;
    }

    public void shutdown() {
        GatewayState.getInstance().lockForUpdating();
        //try
        //{
        // place holder for shutdown code
        //}
        //finally
        //{
        GatewayState.getInstance().unlockFromUpdating();
        //}
    }

    /**
     * Establishes a connection with a SQL server database instance and data
     * source, specified by parameters
     */
    private Connection databaseConnect(String db_hostname, String db_port, String db_name, String db_instance, String db_userid, String db_password) throws SQLException {
        Connection conn = null;
        String db_connect_string = "jdbc:sqlserver://" + db_hostname + ":" + db_port + ";databaseName=" + db_name + ";ssl=require;hostNameInCertificate=" + db_hostname + ";instanceName=" + db_instance + ";portNumber=" + db_port + ";trustServerCertificate=true";

        // Load DB Driver
        try {
            Class.forName("com.microsoft.sqlserver.jdbc.SQLServerDriver");
        } catch (ClassNotFoundException f) {
            LOGGER.trace("Could not load SQL Server DB driver");
        }

        // Connect to DB
        try {
            conn = DriverManager.getConnection(db_connect_string, db_userid, db_password);
        } catch (SQLException sqlException) {
            LOGGER.trace("SQL Server database not reachable");
            throw sqlException;
        }

        return conn;
    }
   
    
    private boolean hasOnlyLocalRecipients(Mail mail) {
        Map<String, String> properties = MailetProperties.getPropertiesList();
        String domainsProperty = properties.get("mailet.domain");
        String[] domainsList = domainsProperty.split(",");
        boolean onlyLocal = true;
        try {
            MimeMessage message = mail.getMessage();
            Address[] recipsAddr = message.getAllRecipients();
            
            if (recipsAddr != null) {
                // escape string for JSON
                for (Address addr : recipsAddr) {
	                boolean isLocal = false;
	                if (recipsAddr != null) {
	                    InternetAddress inet_addr_recip = (InternetAddress) addr;
	                    int atPosition = inet_addr_recip.getAddress().indexOf("@") + 1;
	                    String currentdomain = inet_addr_recip.getAddress().substring(atPosition);
	                    for (int c = 0; c < domainsList.length; c++) {
	                        if (currentdomain.equals(domainsList[c])) {
	                            isLocal = true;
	                        }
	                    }
	                }
	                if (!isLocal) {
	                    onlyLocal = false;
	                } //if any non-local addresses detected, then the recipients are not only local
	            }
            }
        } catch (MessagingException me) {
            LOGGER.debug("Messaging execption while checking for external recipients in recipient list.");
        }
        return onlyLocal;
    }
    
}

